﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Configuration;
using System.Data;
using System.Drawing;
using System.Globalization;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using CRR_Example_Client.ServiceReferences;

namespace CRR_Example_Client
{
    public partial class frmMain : Form
    {
        private string version = "3.0";
        private List<ExecutionLogEntry> log = new List<ExecutionLogEntry>();
        private ProtocolType protocol = ProtocolType.HTTPS;
        private string address = "https://demo-crrservice.mypos.com/infromhttp";
        Configuration config;

        public frmMain()
        {
            InitializeComponent();

            cmbAction.Items.Add(new IdNamePair(1, "Pair"));
            cmbAction.Items.Add(new IdNamePair(2, "Unpair"));
            cmbAction.SelectedIndex = 0;

            cmbReferenceNumberType.Items.Add(new IdNamePair(0, "None"));
            cmbReferenceNumberType.Items.Add(new IdNamePair(1, "Reference Number"));
            cmbReferenceNumberType.Items.Add(new IdNamePair(2, "Invoice Number"));
            cmbReferenceNumberType.Items.Add(new IdNamePair(3, "Product ID"));
            cmbReferenceNumberType.Items.Add(new IdNamePair(4, "Reservation Number"));
            cmbReferenceNumberType.SelectedIndex = 0;

            executionLogEntryBindingSource.DataSource = log;
            executionLogEntryBindingSource.ResetBindings(false);
        }

        private void frmMain_Load(object sender, EventArgs e)
        {
            this.Text = "CRR Example Client " + version.ToString();
            config = ConfigurationManager.OpenExeConfiguration(Application.ExecutablePath);
            LoadConfig();
        }

        private void LoadConfig()
        {
            // Load main form fields
            string last_login = config.AppSettings.Settings["last_login"]?.Value;
            if (last_login != null) txtLogin.Text = last_login;

            string last_key_index = config.AppSettings.Settings["last_key_index"]?.Value;
            if (last_key_index != null) txtKeyIndex.Text = last_key_index;

            string last_tid = config.AppSettings.Settings["last_tid"]?.Value;
            if (last_tid != null) txtTID.Text = last_tid;

            // load communication settings
            string last_protocol = config.AppSettings.Settings["last_protocol"]?.Value;
            if (last_protocol != null) Enum.TryParse(last_protocol, out protocol);

            string last_address = config.AppSettings.Settings["last_address"]?.Value;
            if (last_address != null) address = last_address;

            // load private key
            string last_private_key_pem = config.AppSettings.Settings["last_private_key_pem"]?.Value;
            string last_private_key_mod = config.AppSettings.Settings["last_private_key_mod"]?.Value;
            string last_private_key_exp = config.AppSettings.Settings["last_private_key_exp"]?.Value;
            string last_private_key_D = config.AppSettings.Settings["last_private_key_D"]?.Value;
            string last_private_key_P = config.AppSettings.Settings["last_private_key_P"]?.Value;
            string last_private_key_Q = config.AppSettings.Settings["last_private_key_Q"]?.Value;
            string last_private_key_DP = config.AppSettings.Settings["last_private_key_DP"]?.Value;
            string last_private_key_DQ = config.AppSettings.Settings["last_private_key_DQ"]?.Value;
            string last_private_key_IQ = config.AppSettings.Settings["last_private_key_IQ"]?.Value;

            if (last_private_key_pem != null
                && last_private_key_mod != null
                && last_private_key_exp != null
                && last_private_key_D   != null
                && last_private_key_P   != null
                && last_private_key_Q   != null
                && last_private_key_DP  != null
                && last_private_key_DQ  != null
                && last_private_key_IQ  != null
                )
            {
                int private_key_size = Math.Max(last_private_key_mod.Length, last_private_key_D.Length) * 4;

                CspParameters cp = new CspParameters();
                cp.Flags = CspProviderFlags.UseMachineKeyStore;
                cp.KeyContainerName = "CRRKeys";
                RSACryptoServiceProvider PrivateKey = new RSACryptoServiceProvider(private_key_size, cp);

                RSAParameters RSAparams = new RSAParameters();

                RSAparams.Modulus = Misc.HEXToByteArray(last_private_key_mod);
                RSAparams.Exponent = Misc.HEXToByteArray(last_private_key_exp);
                RSAparams.D = Misc.HEXToByteArray(last_private_key_D);
                RSAparams.P = Misc.HEXToByteArray(last_private_key_P);
                RSAparams.Q = Misc.HEXToByteArray(last_private_key_Q);
                RSAparams.DP = Misc.HEXToByteArray(last_private_key_DP);
                RSAparams.DQ = Misc.HEXToByteArray(last_private_key_DQ);
                RSAparams.InverseQ = Misc.HEXToByteArray(last_private_key_IQ);

                try
                {
                    PrivateKey.ImportParameters(RSAparams);
                    SecurityRSA.PrivateKeyForSigningRequests = PrivateKey;
                    SecurityRSA.OurPrivateKeyPEM = last_private_key_pem;
                }
                catch //(Exception ex)
                {
                }
            }

            // load public key
            string last_public_key_pem = config.AppSettings.Settings["last_public_key_pem"]?.Value;
            string last_public_key_mod = config.AppSettings.Settings["last_public_key_mod"]?.Value;
            string last_public_key_exp = config.AppSettings.Settings["last_public_key_exp"]?.Value;

            if (last_public_key_pem != null
                && last_public_key_mod != null
                && last_public_key_exp != null
                )
            {
                int public_key_size = Math.Max(last_public_key_mod.Length, last_public_key_exp.Length) * 4;

                CspParameters cp = new CspParameters();
                cp.Flags = CspProviderFlags.UseMachineKeyStore;
                cp.KeyContainerName = "CRRKeys";
                RSACryptoServiceProvider PublicKey = new RSACryptoServiceProvider(public_key_size, cp);

                RSAParameters RSAparams = new RSAParameters();

                RSAparams.Modulus = Misc.HEXToByteArray(last_private_key_mod);
                RSAparams.Exponent = Misc.HEXToByteArray(last_private_key_exp);

                try
                {
                    PublicKey.ImportParameters(RSAparams);
                    SecurityRSA.PublicKeyForValidatingAnswers = PublicKey;
                    SecurityRSA.TheirPublicCertificatePEM = last_public_key_pem;
                }
                catch //(Exception ex)
                {
                }
            }

        }

        private void SaveMainFormConfig()
        {
            config.AppSettings.Settings.Remove("last_login");
            config.AppSettings.Settings.Remove("last_key_index");
            config.AppSettings.Settings.Remove("last_tid");

            config.AppSettings.Settings.Add("last_login", txtLogin.Text);
            config.AppSettings.Settings.Add("last_key_index", txtKeyIndex.Text);
            config.AppSettings.Settings.Add("last_tid", txtTID.Text);

            config.Save();
        }

        private void SaveCommunicationConfig()
        {
            config.AppSettings.Settings.Remove("last_protocol");
            config.AppSettings.Settings.Remove("last_address");

            config.AppSettings.Settings.Add("last_protocol", protocol.ToString());
            config.AppSettings.Settings.Add("last_address", address);

            config.Save();
        }

        private void SaveKeysConfig()
        {
            config.AppSettings.Settings.Remove("last_private_key_pem");
            config.AppSettings.Settings.Remove("last_private_key_mod");
            config.AppSettings.Settings.Remove("last_private_key_exp");
            config.AppSettings.Settings.Remove("last_private_key_D");
            config.AppSettings.Settings.Remove("last_private_key_P");
            config.AppSettings.Settings.Remove("last_private_key_Q");
            config.AppSettings.Settings.Remove("last_private_key_DP");
            config.AppSettings.Settings.Remove("last_private_key_DQ");
            config.AppSettings.Settings.Remove("last_private_key_IQ");

            config.AppSettings.Settings.Remove("last_public_key_pem");
            config.AppSettings.Settings.Remove("last_public_key_mod");
            config.AppSettings.Settings.Remove("last_public_key_exp");

            RSAParameters private_params = SecurityRSA.PrivateKeyForSigningRequests.ExportParameters(true);
            RSAParameters public_params = SecurityRSA.PublicKeyForValidatingAnswers.ExportParameters(false);

            config.AppSettings.Settings.Add("last_private_key_pem", SecurityRSA.OurPrivateKeyPEM);
            config.AppSettings.Settings.Add("last_private_key_mod", Misc.ByteArrayToHEX(private_params.Modulus));
            config.AppSettings.Settings.Add("last_private_key_exp", Misc.ByteArrayToHEX(private_params.Exponent));
            config.AppSettings.Settings.Add("last_private_key_D", Misc.ByteArrayToHEX(private_params.D));
            config.AppSettings.Settings.Add("last_private_key_P", Misc.ByteArrayToHEX(private_params.P));
            config.AppSettings.Settings.Add("last_private_key_Q", Misc.ByteArrayToHEX(private_params.Q));
            config.AppSettings.Settings.Add("last_private_key_DP", Misc.ByteArrayToHEX(private_params.DP));
            config.AppSettings.Settings.Add("last_private_key_DQ", Misc.ByteArrayToHEX(private_params.DQ));
            config.AppSettings.Settings.Add("last_private_key_IQ", Misc.ByteArrayToHEX(private_params.InverseQ));

            config.AppSettings.Settings.Add("last_public_key_pem", SecurityRSA.TheirPublicCertificatePEM);
            config.AppSettings.Settings.Add("last_public_key_mod", Misc.ByteArrayToHEX(public_params.Modulus));
            config.AppSettings.Settings.Add("last_public_key_exp", Misc.ByteArrayToHEX(public_params.Exponent));

            config.Save();
        }

        string ConcatExceptionMessage(Exception ex)
        {
            string result = "";
            Exception pos = ex;
            while (pos != null)
            {
                result += pos.Message;
                pos = pos.InnerException;
            }
            if (result == "")
            {
                return null;
            }
            return result;
        }

        private string GetBindingNameFromProtocol(ProtocolType p)
        {
            switch (p)
            {
                case ProtocolType.HTTP:
                    return "BasicHttpBinding_IiPayCRRRequest";
                case ProtocolType.HTTPS:
                    return "HTTPSBinding_IiPayCRRRequest";
                default:
                    return "BasicHttpBinding_IiPayCRRRequest";
            }
        }

        private void AddLogEntry(ExecutionLogEntry le)
        {
            le.dtEntry = DateTime.Now;
            log.Add(le);

            executionLogEntryBindingSource.ResetBindings(false);
            gridLog.ClearSelection();
            int row = gridLog.Rows.GetLastRow(DataGridViewElementStates.None);
            if (row >= 0)
            {
                gridLog.Rows[row].Selected = true;
            }
        }

        private void RefereshLog(ExecutionLogEntry le, iCResult res, bool? signature_match = null)
        {
            if (le == null) return;
            le.result = res;
            le.time = (DateTime.Now - le.dtEntry).TotalSeconds.ToString("F3") + " seconds";
            le.signature_match = signature_match;

            executionLogEntryBindingSource.ResetBindings(false);
            gridLog.ClearSelection();
            int row = gridLog.Rows.GetLastRow(DataGridViewElementStates.None);
            if (row >= 0)
            {
                gridLog.Rows[row].Selected = true;
            }

            propertyGridLogEntry.SelectedObject = null;
            propertyGridLogEntry.SelectedObject = le.result;
        }
        private void gridLog_SelectionChanged(object sender, EventArgs e)
        {
            if (gridLog.CurrentCell == null) return;
            int row = gridLog.CurrentCell.RowIndex;
            if (row < 0) return;

            ExecutionLogEntry le = gridLog.Rows[0].DataBoundItem as ExecutionLogEntry;
            if (le == null) return;

            if (propertyGridLogEntry.SelectedObject == le) return;

            propertyGridLogEntry.SelectedObject = null;
            propertyGridLogEntry.SelectedObject = le.result;
        }

        private void txtAutoRUID_Click(object sender, EventArgs e)
        {
            txtRUID.Text = DateTime.Now.ToString("yyyyMMddHHmmssfff") + new Random().Next(1000).ToString("D3");
        }

        private void connectionSettingsToolStripMenuItem_Click(object sender, EventArgs e)
        {
            frmConnectionEdit cf = new frmConnectionEdit(protocol, address);
            if (cf.ShowDialog() == System.Windows.Forms.DialogResult.OK)
            {
                protocol = cf.GetProtocol();
                address = cf.GetAddress();
                SaveCommunicationConfig();
            }
        }

        private void keyManagementToolStripMenuItem_Click(object sender, EventArgs e)
        {
            frmKeyManagement fKM = new frmKeyManagement();
            if (fKM.ShowDialog() == System.Windows.Forms.DialogResult.OK)
            {
                SaveKeysConfig();
            }
        }

        private void btnCopyRUID_Click(object sender, EventArgs e)
        {
            txtRUIDOriginal.Text = txtRUID.Text;
        }

        private void btnSubscribe_Click(object sender, EventArgs e)
        {
            iCMPRSubscribeResult res = null;
            bool? signature_match = null;

            string login = txtLogin.Text;
            int key_index;
            if (!Int32.TryParse(txtKeyIndex.Text, out key_index)) return;
            string ruid = txtRUID.Text;
            string tid = txtTID.Text;
            string currency = txtCurrency.Text;
            int action = ((IdNamePair)(cmbAction.SelectedItem)).id;

            ExecutionLogEntry le = new ExecutionLogEntry();
            le.strCommand = "MPRSubscribe";
            AddLogEntry(le);

            try
            {
                SaveMainFormConfig();

                string string_to_sign = string.Format(CultureInfo.InvariantCulture
                    , "{0};{1};{2};{3};{4};{5};{6}"
                    , key_index, version, login, tid, ruid, currency, action
                    );

                byte[] DataToSign = System.Text.UTF8Encoding.UTF8.GetBytes(string_to_sign);
                byte[] Hash_Signature = SecurityRSA.HashAndSignData(DataToSign);

                IiPayCRRRequestClient req;
                if (address.Length > 0)
                {
                    req = new IiPayCRRRequestClient(GetBindingNameFromProtocol(protocol), address);
                }
                else
                {
                    req = new IiPayCRRRequestClient(GetBindingNameFromProtocol(protocol));
                }

                res = req.MPRSubscribe(version, login, key_index, tid, ruid, currency, action, Hash_Signature);

                String string_to_verify = String.Format(
                    "{0};{1};{2};{3};{4};{5};{6};{7};{8};{9}"
                    , res.key_index, res.version, res.login, res.tid, res.ruid, res.currency, res.action
                    , res.code, res.status, res.status_msg
                    );
                byte[] DataToVerify = System.Text.UTF8Encoding.UTF8.GetBytes(string_to_verify);

                signature_match = SecurityRSA.HashAndVerifyData(DataToVerify, res.signature);
            }
            catch (Exception ex)
            {
                le.exception_text = ConcatExceptionMessage(ex);
            }

            RefereshLog(le, res, signature_match);
        }

        private void btnPurchase_Click(object sender, EventArgs e)
        {
			iCMPRPurchaseResult res = null;
			bool? signature_match = null;

			string login = txtLogin.Text;
			int key_index;
			if (!Int32.TryParse(txtKeyIndex.Text, out key_index)) return;
			string tid = txtTID.Text;
			string ruid = txtRUID.Text;
			string amount = txtAmount.Text;
			string reference_number = txtReferenceNumber.Text;
			int reference_number_type = ((IdNamePair)cmbReferenceNumberType.SelectedItem).id;

			ExecutionLogEntry le = new ExecutionLogEntry();
			le.strCommand = "MPRPurchase";
			AddLogEntry(le);

			try
            {
                SaveMainFormConfig();

                string string_to_sign = string.Format(CultureInfo.InvariantCulture
                    , "{0};{1};{2};{3};{4};{5};{6};{7}"
                    , key_index, version, login, tid, ruid, amount, reference_number, reference_number_type
                );

                byte[] DataToSign = System.Text.UTF8Encoding.UTF8.GetBytes(string_to_sign);
                byte[] Hash_Signature = SecurityRSA.HashAndSignData(DataToSign);

                IiPayCRRRequestClient req;
				if (address.Length > 0)
				{
					req = new IiPayCRRRequestClient(GetBindingNameFromProtocol(protocol), address);
				}
				else
				{
					req = new IiPayCRRRequestClient(GetBindingNameFromProtocol(protocol));
				}
				res = req.MPRPurchase(version, login, key_index, tid, ruid, amount, reference_number, reference_number_type, Hash_Signature);

				String string_to_verify = String.Format(
					"{0};{1};{2};{3};{4};{5};{6};{7};{8};{9}"
					, res.key_index, res.version, res.login, res.tid, res.ruid, res.amount, res.reference_number, res.reference_number_type
					, res.status, res.status_msg
					);

                byte[] DataToVerify = System.Text.UTF8Encoding.UTF8.GetBytes(string_to_verify);
                signature_match = SecurityRSA.HashAndVerifyData(DataToVerify, res.signature);
            }
			catch (Exception ex)
			{
				le.exception_text = ConcatExceptionMessage(ex);
			}

			RefereshLog(le, res, signature_match);
        }

        private void btnRefund_Click(object sender, EventArgs e)
        {
            iCMPRRefundResult res = null;
            bool? signature_match = null;

            string login = txtLogin.Text;
            int key_index;
            if (!Int32.TryParse(txtKeyIndex.Text, out key_index)) return;
            string tid = txtTID.Text;
            string ruid = txtRUID.Text;
            string amount = txtAmount.Text;

            ExecutionLogEntry le = new ExecutionLogEntry();
            le.strCommand = "MPRRefund";
            AddLogEntry(le);

            try
            {
                SaveMainFormConfig();

                string string_to_sign = string.Format(CultureInfo.InvariantCulture
                    , "{0};{1};{2};{3};{4};{5}"
                    , key_index, version, login, tid, ruid, amount
                );

                byte[] DataToSign = System.Text.UTF8Encoding.UTF8.GetBytes(string_to_sign);
                byte[] Hash_Signature = SecurityRSA.HashAndSignData(DataToSign);

                IiPayCRRRequestClient req;
                if (address.Length > 0)
                {
                    req = new IiPayCRRRequestClient(GetBindingNameFromProtocol(protocol), address);
                }
                else
                {
                    req = new IiPayCRRRequestClient(GetBindingNameFromProtocol(protocol));
                }
                res = req.MPRRefund(version, login, key_index, tid, ruid, amount, Hash_Signature);

                String string_to_verify = String.Format(
                    "{0};{1};{2};{3};{4};{5};{6};{7}"
                    , res.key_index, res.version, res.login, res.tid, res.ruid, res.amount
                    , res.status, res.status_msg
                    );

                byte[] DataToVerify = System.Text.UTF8Encoding.UTF8.GetBytes(string_to_verify);
                signature_match = SecurityRSA.HashAndVerifyData(DataToVerify, res.signature);
            }
            catch (Exception ex)
            {
                le.exception_text = ConcatExceptionMessage(ex);
            }

            RefereshLog(le, res, signature_match);
        }

        private void btnVoid_Click(object sender, EventArgs e)
        {
            iCMPRLastTxnVoidResult res = null;
            bool? signature_match = null;

            string login = txtLogin.Text;
            int key_index;
            if (!Int32.TryParse(txtKeyIndex.Text, out key_index)) return;
            string tid = txtTID.Text;
            string ruid = txtRUID.Text;
            string ruid_original = txtRUIDOriginal.Text;
            string amount = txtAmount.Text;

            ExecutionLogEntry le = new ExecutionLogEntry();
            le.strCommand = "MPRLastTxnVoid";
            AddLogEntry(le);

            try
            {
                SaveMainFormConfig();

                string string_to_sign = string.Format(CultureInfo.InvariantCulture
                    , "{0};{1};{2};{3};{4};{5};{6}"
                    , key_index, version, login, tid, ruid, ruid_original, amount
                );

                byte[] DataToSign = System.Text.UTF8Encoding.UTF8.GetBytes(string_to_sign);
                byte[] Hash_Signature = SecurityRSA.HashAndSignData(DataToSign);

                IiPayCRRRequestClient req;
                if (address.Length > 0)
                {
                    req = new IiPayCRRRequestClient(GetBindingNameFromProtocol(protocol), address);
                }
                else
                {
                    req = new IiPayCRRRequestClient(GetBindingNameFromProtocol(protocol));
                }
                res = req.MPRLastTxnVoid(version, login, key_index, tid, ruid, ruid_original, amount, Hash_Signature);

                String string_to_verify = String.Format(
                    "{0};{1};{2};{3};{4};{5};{6};{7};{8}"
                    , res.key_index, res.version, res.login, res.tid, res.ruid, res.ruid_original, res.amount
                    , res.status, res.status_msg
                    );

                byte[] DataToVerify = System.Text.UTF8Encoding.UTF8.GetBytes(string_to_verify);
                signature_match = SecurityRSA.HashAndVerifyData(DataToVerify, res.signature);
            }
            catch (Exception ex)
            {
                le.exception_text = ConcatExceptionMessage(ex);
            }

            RefereshLog(le, res, signature_match);
        }

        private void btnTranStatus_Click(object sender, EventArgs e)
        {
            iCMPRGetTxnStatusResult res = null;
            bool? signature_match = null;

            string login = txtLogin.Text;
            int key_index;
            if (!Int32.TryParse(txtKeyIndex.Text, out key_index)) return;
            string tid = txtTID.Text;
            string ruid = txtRUID.Text;
            string ruid_original = txtRUIDOriginal.Text;

            ExecutionLogEntry le = new ExecutionLogEntry();
            le.strCommand = "MPRGetTxnStatus";
            AddLogEntry(le);

            try
            {
                SaveMainFormConfig();

                string string_to_sign = string.Format(CultureInfo.InvariantCulture
                    , "{0};{1};{2};{3};{4};{5}"
                    , key_index, version, login, tid, ruid, ruid_original, ruid_original
                );

                byte[] DataToSign = System.Text.UTF8Encoding.UTF8.GetBytes(string_to_sign);
                byte[] Hash_Signature = SecurityRSA.HashAndSignData(DataToSign);

                IiPayCRRRequestClient req;
                if (address.Length > 0)
                {
                    req = new IiPayCRRRequestClient(GetBindingNameFromProtocol(protocol), address);
                }
                else
                {
                    req = new IiPayCRRRequestClient(GetBindingNameFromProtocol(protocol));
                }
                res = req.MPRGetTxnStatus(version, login, key_index, tid, ruid, ruid_original, Hash_Signature);

                String string_to_verify = String.Format(
                    "{0};{1};{2};{3};{4};{5};{6};{7};{8};{9};{10};{11};{12};{13};{14};{15};{16};{17};{18};{19};{20};{21};{22};{23};{24};{25};{26};{27};{28}"
                    , res.key_index, res.version, res.login, res.tid, res.ruid, res.ruid_original
                    , res.ruid_original_status, res.original_method, res.amount
                    , res.response_code, res.auth_code, res.rrn, res.stan, res.merch_txn_date, res.merch_txn_time
                    , res.mid, res.merch_name, res.merch_address_line1, res.merch_address_line2
                    , res.masked_pan, res.emboss_name, res.aid, res.aid_name, res.signature_required
                    , res.reference_number, res.reference_number_type
                    , res.payment_reference
                    , res.status, res.status_msg
                    );


                byte[] DataToVerify = System.Text.UTF8Encoding.UTF8.GetBytes(string_to_verify);
                signature_match = SecurityRSA.HashAndVerifyData(DataToVerify, res.signature);
            }
            catch (Exception ex)
            {
                le.exception_text = ConcatExceptionMessage(ex);
            }

            RefereshLog(le, res, signature_match);
        }

        private void btnCancel_Click(object sender, EventArgs e)
        {
            iCMPRCancelResult res = null;
            bool? signature_match = null;

            string login = txtLogin.Text;
            int key_index;
            if (!Int32.TryParse(txtKeyIndex.Text, out key_index)) return;
            string tid = txtTID.Text;
            string ruid = txtRUID.Text;
            string ruid_original = txtRUIDOriginal.Text;

            ExecutionLogEntry le = new ExecutionLogEntry();
            le.strCommand = "MPRCancel";
            AddLogEntry(le);

            try
            {
                SaveMainFormConfig();

                string string_to_sign = string.Format(CultureInfo.InvariantCulture
                    , "{0};{1};{2};{3};{4};{5}"
                    , key_index, version, login, tid, ruid, ruid_original, ruid_original
                );

                byte[] DataToSign = System.Text.UTF8Encoding.UTF8.GetBytes(string_to_sign);
                byte[] Hash_Signature = SecurityRSA.HashAndSignData(DataToSign);

                IiPayCRRRequestClient req;
                if (address.Length > 0)
                {
                    req = new IiPayCRRRequestClient(GetBindingNameFromProtocol(protocol), address);
                }
                else
                {
                    req = new IiPayCRRRequestClient(GetBindingNameFromProtocol(protocol));
                }
                res = req.MPRCancel(version, login, key_index, tid, ruid, ruid_original, Hash_Signature);

                String string_to_verify = String.Format(
                    "{0};{1};{2};{3};{4};{5};{6};{7}"
                    , res.key_index, res.version, res.login, res.tid, res.ruid, res.ruid_original
                    , res.status, res.status_msg
                    );

                byte[] DataToVerify = System.Text.UTF8Encoding.UTF8.GetBytes(string_to_verify);
                signature_match = SecurityRSA.HashAndVerifyData(DataToVerify, res.signature);
            }
            catch (Exception ex)
            {
                le.exception_text = ConcatExceptionMessage(ex);
            }

            RefereshLog(le, res, signature_match);
        }
    }
}
